<?php
session_start();

/* =========================================================
   1) CONFIGURAÇÕES E CONEXÕES DE BANCO
   ========================================================= */
define('DNS_DB_PATH', __DIR__ . '/api/db/studiolivecode_dns.db');       // Para preencher o <select> de DNS
define('CLIENT_DB_PATH', __DIR__ . '/api/db/studiolivecode_clientes.db'); // Tabela que guarda os dispositivos (ibo)
define('USERS_DB_PATH', __DIR__ . '/api/db/studiolivecode_users.db');     // Tabela de usuários

function sanitize($str) {
    return htmlspecialchars(trim($str), ENT_QUOTES, 'UTF-8');
}

// Conecta ao banco DNS para obter as opções (title)
$dnsOptions = [];
try {
    $connDns = new PDO('sqlite:' . DNS_DB_PATH);
    $connDns->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    $stmt = $connDns->query("SELECT title FROM dns");
    $dnsOptions = $stmt->fetchAll(PDO::FETCH_COLUMN);
} catch (PDOException $e) {
    echo "Erro ao conectar ao banco DNS: " . $e->getMessage();
    exit();
}

/* =========================================================
   2) TRATAMENTO DE AÇÕES VIA AJAX
   ========================================================= */
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
    $action = $_POST['action'];

    // ---------- 2.1) Verificar se a key (ID) já existe -----------
    if ($action === 'check_key') {
        $deviceKey = sanitize($_POST['device_key'] ?? '');
        if ($deviceKey === '') {
            echo json_encode(['status' => 'error', 'message' => 'Chave vazia']);
            exit();
        }
        // Consulta no banco CLIENTES (tabela ibo)
        try {
            $pdo = new PDO('sqlite:' . CLIENT_DB_PATH);
            $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

            // Verifica se existe algum registro com essa key
            $stmt = $pdo->prepare("SELECT * FROM ibo WHERE key = :key LIMIT 1");
            $stmt->bindValue(':key', $deviceKey);
            $stmt->execute();
            $device = $stmt->fetch(PDO::FETCH_ASSOC);

            if ($device) {
                // Já existe -> status: found
                echo json_encode(['status' => 'found']);
            } else {
                // Não existe -> status: not_found
                echo json_encode(['status' => 'not_found']);
            }
        } catch (PDOException $e) {
            echo json_encode(['status' => 'error', 'message' => $e->getMessage()]);
        }
        exit();
    }

    // ---------- 2.2) Se o usuário optar por "Editar", puxar dados ----------
    if ($action === 'fetch_device') {
        $deviceKey = sanitize($_POST['device_key'] ?? '');
        if ($deviceKey === '') {
            echo json_encode(['status' => 'error', 'message' => 'Chave vazia']);
            exit();
        }
        try {
            $pdo = new PDO('sqlite:' . CLIENT_DB_PATH);
            $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

            $stmt = $pdo->prepare("SELECT * FROM ibo WHERE key = :key LIMIT 1");
            $stmt->bindValue(':key', $deviceKey);
            $stmt->execute();
            $device = $stmt->fetch(PDO::FETCH_ASSOC);

            if ($device) {
                // Supondo colunas: key, dns, user_id, nomePlaylist, username, password
                echo json_encode([
                    'status'    => 'success',
                    'dns'       => $device['dns'],
                    'nome_list' => $device['title'] ?? '',     // ou 'nomePlaylist', se preferir
                    'user_id'   => $device['user_id'] ?? '',
                    'username'  => $device['username'] ?? '',
                    'password'  => $device['password'] ?? ''
                ]);
            } else {
                echo json_encode(['status' => 'not_found']);
            }
        } catch (PDOException $e) {
            echo json_encode(['status' => 'error', 'message' => $e->getMessage()]);
        }
        exit();
    }

    // ---------- 2.3) Ao digitar o NOME do usuário, buscar seu ID no DB de usuários ----------
    if ($action === 'fetch_user_id') {
        $userName = sanitize($_POST['username'] ?? '');
        if ($userName === '') {
            echo json_encode(['status' => 'error', 'message' => 'Nome de usuário vazio']);
            exit();
        }
        try {
            $pdo = new PDO('sqlite:' . USERS_DB_PATH);
            $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

            // Exemplo: se a tabela users tem colunas [id, name, ...]
            $stmt = $pdo->prepare("SELECT id FROM users WHERE name = :name LIMIT 1");
            $stmt->bindValue(':name', $userName);
            $stmt->execute();
            $userRow = $stmt->fetch(PDO::FETCH_ASSOC);

            if ($userRow) {
                // Encontrou um ID
                echo json_encode(['status' => 'success', 'user_id' => $userRow['id']]);
            } else {
                // Não encontrou
                echo json_encode(['status' => 'not_found']);
            }
        } catch (PDOException $e) {
            echo json_encode(['status' => 'error', 'message' => $e->getMessage()]);
        }
        exit();
    }
    // Se chegou aqui, finaliza
    exit();
}

/* =========================================================
   3) HTML E JAVASCRIPT
   ========================================================= */
?>
<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8" />
  <meta name="viewport" content="width=device-width,initial-scale=1.0" />
  <title>Adicionar/Editar Playlist</title>
  <style>
    body {
      font-family: Arial, sans-serif;
      margin: 0;
      padding: 0 10px;
      background: url('https://source.unsplash.com/1600x900/?programming') center/cover no-repeat fixed;
    }
    #container {
      display: flex;
      justify-content: center;
      align-items: center;
      height: 100vh;
    }
    #form-container {
      background-color: rgba(255, 255, 255, 0.8);
      padding: 20px;
      border-top: 5px solid #0A38D1;
      border-radius: 10px;
      box-shadow: 0 0 10px rgba(0, 0, 0, 0.1);
      max-width: 400px;
      width: 100%;
      text-align: center;
    }
    h2 {
      text-align: center;
      margin-bottom: 20px;
    }
    label {
      display: block;
      text-align: left;
      margin-top: 5px;
      margin-left: 20px;
    }
    input, select {
      width: calc(100% - 40px);
      padding: 15px 10px;
      margin-top: 5px;
      margin-bottom: 15px;
      box-sizing: border-box;
      border-radius: 5px;
      border: 1px solid #ccc;
    }
    .input-group {
      position: relative;
      width: calc(100% - 40px);
      margin: 0 auto 15px;
    }
    .input-group input {
      width: 100%;
      padding-right: 40px;
    }
    .input-group .toggle-password {
      position: absolute;
      right: 10px;
      top: 50%;
      transform: translateY(-80%);
      cursor: pointer;
      color: #0A38D1;
    }
    .input-group .toggle-password i {
      font-size: 18px; 
    }
    button {
      width: 100%;
      margin-top: 10px;
      padding: 15px 10px;
      border-radius: 5px;
      background-color: #0A38D1;
      color: #ffffff;
      border: none;
      cursor: pointer;
    }
    button:hover {
      background-color: #0529a3;
    }
    #logo {
      max-width: 100%;
      height: auto;
      margin: 0 auto 20px;
    }
    /* Modais */
    #macModal, #successModal, #errorModal, #invalidUserPassModal {
      display: none;
      position: fixed;
      z-index: 1;
      left: 0;
      top: 0;
      width: 100%;
      height: 100%;
      overflow: auto;
      background-color: rgba(0, 0, 0, 0.4);
    }
    .modal-content {
      background-color: #fefefe;
      margin: 15% auto;
      padding: 20px;
      border: 1px solid #888;
      width: 80%;
      max-width: 300px;
      text-align: center;
      border-radius: 10px;
    }
    .close {
      color: #aaa;
      float: right;
      font-size: 28px;
      font-weight: bold;
    }
    .close:hover,
    .close:focus {
      color: black;
      text-decoration: none;
      cursor: pointer;
    }
  </style>
</head>
<body>
  <!-- Modal para digitar a Chave do Dispositivo -->
  <div id="macModal">
    <div class="modal-content">
      <h3>Chave do Dispositivo</h3>
      <input type="text" id="modalIdInput" placeholder="Digite a Chave do Dispositivo">
      <button onclick="verificarId()">Confirmar</button>
    </div>
  </div>

  <!-- Modal de Sucesso -->
  <div id="successModal">
    <div class="modal-content">
      <span class="close" onclick="closeSuccessModal()">&times;</span>
      <h2>Sucesso!</h2>
      <p>Dados inseridos com sucesso. Redirecionando...</p>
    </div>
  </div>

  <!-- Modal de Erro (ID Inválido) -->
  <div id="errorModal">
    <div class="modal-content">
      <span class="close" onclick="closeErrorModal()">&times;</span>
      <h2>Erro!</h2>
      <p>ID inválido. Tente novamente.</p>
    </div>
  </div>

  <!-- Modal de Erro (Usuário ou senha) -->
  <div id="invalidUserPassModal">
    <div class="modal-content">
      <span class="close" onclick="closeInvalidUserPassModal()">&times;</span>
      <h2>Erro!</h2>
      <p>Usuário ou senha está errado. Tente novamente.</p>
    </div>
  </div>

  <!-- Conteúdo Principal -->
  <div id="container">
    <div id="form-container">
      <h2>Adicionar/Editar Dispositivo</h2>
      <img id="logo" style="width:60px" src="./assets/images/logo.png" alt="Logo">

      <!-- Formulário principal que vai para authqr.php -->
      <form method="post" action="authqr.php" id="loginForm">
        <!-- DNS -->
        <label for="dns">Conexão:</label>
        <select id="dns" name="dns" required>
          <?php
          foreach ($dnsOptions as $index => $dnsOption) {
              $dnsOptionEsc = htmlspecialchars($dnsOption, ENT_QUOTES, 'UTF-8');
              $selected = ($index === 0) ? 'selected' : '';
              echo "<option value=\"$dnsOptionEsc\" $selected>$dnsOptionEsc</option>";
          }
          ?>
        </select>

        <!-- ID (Device Key) -->
        <input type="hidden" id="id" name="id" required>

        <!-- Nome da Lista -->
        <label for="nomePlaylist">Nome da Lista:</label>
        <input type="text" id="nomePlaylist" name="nomePlaylist" placeholder="Digite o nome da lista" required>

        <!-- (Campo hidden que conterá o user_id real) -->
        <input type="hidden" id="hidden_user_id" name="hidden_user_id">

        <!-- ID do Usuário (mas na verdade pediremos o NOME) -->
        <label for="user_id">USER REVENDA:</label>
        <input type="text" id="user_id" name="user_id" placeholder="Digite o NOME do usuário" required>

        <!-- Usuário -->
        <label for="usuario">Usuário:</label>
        <input type="text" id="usuario" name="usuario" placeholder="Digite o seu usuário" required>

        <!-- Senha -->
        <label for="senha">Senha:</label>
        <div class="input-group">
          <input type="password" id="senha" name="senha" placeholder="Digite a sua senha" required>
          <span class="toggle-password" onclick="toggleSenha()">
            <i class="fa fa-eye-slash" id="toggleIcon"></i>
          </span>
        </div>

        <!-- Botão de enviar -->
        <button class="input-group" type="submit">Enviar</button>
      </form>
      <label for="mac" id="statusMessage"></label>
    </div>
  </div>

  <!-- Scripts -->
  <script>
    // Fecha Modal (Key)
    function closeMacModal() {
      document.getElementById('macModal').style.display = 'none';
    }
    // Fecha Modal de sucesso
    function closeSuccessModal() {
      document.getElementById('successModal').style.display = 'none';
    }
    // Fecha Modal de erro (ID)
    function closeErrorModal() {
      document.getElementById('errorModal').style.display = 'none';
    }
    // Fecha Modal de erro (usuário/senha)
    function closeInvalidUserPassModal() {
      document.getElementById('invalidUserPassModal').style.display = 'none';
    }
    // Mostrar modal de erro (ID)
    function showErrorModal() {
      document.getElementById('errorModal').style.display = 'block';
    }
    // Mostrar modal de erro (usuário/senha)
    function showInvalidUserPassModal() {
      document.getElementById('invalidUserPassModal').style.display = 'block';
    }

    // Mostrar/ocultar senha
    function toggleSenha() {
      var senha = document.getElementById('senha');
      var toggleIcon = document.getElementById('toggleIcon');
      if (senha.type === "password") {
        senha.type = "text";
        toggleIcon.classList.remove('fa-eye-slash');
        toggleIcon.classList.add('fa-eye');
      } else {
        senha.type = "password";
        toggleIcon.classList.remove('fa-eye');
        toggleIcon.classList.add('fa-eye-slash');
      }
    }

    // (1) Ao carregar a página, exibe o modal para digitar a key
    window.onload = function() {
      // Abre o modal para digitar a Key
      document.getElementById('macModal').style.display = 'block';

      <?php if (isset($_SESSION['success_message'])): ?>
        document.getElementById('successModal').style.display = 'block';
        setTimeout(function() {
          window.location.href = "./loginqr.php";
        }, 5000);
        <?php unset($_SESSION['success_message']); ?>
      <?php endif; ?>

      <?php if (isset($_SESSION['error_message'])): ?>
        showInvalidUserPassModal();
        <?php unset($_SESSION['error_message']); ?>
      <?php endif; ?>
    };

    // (2) Verifica se a Key existe
    function verificarId() {
      var keyVal = document.getElementById('modalIdInput').value.trim();
      if (!keyVal) {
        alert("Por favor, digite a chave.");
        return;
      }
      // AJAX para verificar se já existe no banco
      var xhr = new XMLHttpRequest();
      xhr.open("POST", "loginqr.php", true);
      xhr.setRequestHeader("Content-Type", "application/x-www-form-urlencoded");
      xhr.onreadystatechange = function() {
        if (xhr.readyState === 4) {
          if (xhr.status === 200) {
            try {
              var resp = JSON.parse(xhr.responseText);
              if (resp.status === 'found') {
                // Key já existe -> perguntar se quer editar ou criar novo
                if (confirm("Este dispositivo já existe. Deseja EDITAR (OK) ou criar NOVO (Cancelar)?")) {
                  // EDITAR -> fetch_device
                  fetchDeviceData(keyVal);
                } else {
                  // NOVO -> apenas limpa form
                  closeMacModal();
                  document.getElementById('id').value = keyVal;
                  // Limpamos apenas alguns campos
                  document.getElementById('dns').selectedIndex = 0;
                  document.getElementById('nomePlaylist').value = '';
                  document.getElementById('user_id').value = '';
                  document.getElementById('hidden_user_id').value = '';
                  document.getElementById('usuario').value = '';
                  document.getElementById('senha').value = '';
                }
              } else if (resp.status === 'not_found') {
                // Não existe -> OK, criar novo
                closeMacModal();
                document.getElementById('id').value = keyVal;
              } else {
                // Erro
                showErrorModal();
              }
            } catch(e) {
              console.log(e);
              showErrorModal();
            }
          } else {
            showErrorModal();
          }
        }
      };
      xhr.send("action=check_key&device_key=" + encodeURIComponent(keyVal));
    }

    // (3) Se o usuário quiser EDITAR, puxamos do banco e preenchemos
    function fetchDeviceData(keyVal) {
      var xhr = new XMLHttpRequest();
      xhr.open("POST", "loginqr.php", true);
      xhr.setRequestHeader("Content-Type", "application/x-www-form-urlencoded");
      xhr.onreadystatechange = function() {
        if (xhr.readyState === 4) {
          if (xhr.status === 200) {
            try {
              var resp = JSON.parse(xhr.responseText);
              if (resp.status === 'success') {
                closeMacModal();
                // Preenchemos o form
                document.getElementById('id').value = keyVal;
                // DNS
                var dnsSelect = document.getElementById('dns');
                for (var i = 0; i < dnsSelect.options.length; i++) {
                  if (dnsSelect.options[i].value === resp.dns) {
                    dnsSelect.selectedIndex = i;
                    break;
                  }
                }
                // Nome da lista
                document.getElementById('nomePlaylist').value = resp.nome_list || '';
                // user_id (oculto é hidden_user_id). 
                // Você pode colocar o ID real em hidden_user_id e exibir o "nome" no campo user_id
                document.getElementById('hidden_user_id').value = resp.user_id;
                // Caso deseje exibir esse ID no input, ao invés do NOME, você pode alterar a lógica.
                document.getElementById('user_id').value = '[UserNameDesconhecido]'; 
                // <<-- se quiser, precisa de um endpoint extra p/ buscar o "name" do user
                document.getElementById('usuario').value = resp.username || '';
                document.getElementById('senha').value = resp.password || '';
              } else {
                showErrorModal();
              }
            } catch(e) {
              console.log(e);
              showErrorModal();
            }
          } else {
            showErrorModal();
          }
        }
      };
      xhr.send("action=fetch_device&device_key=" + encodeURIComponent(keyVal));
    }

    // (4) Ao digitar o "nome do usuário" no campo #user_id, obtemos o user_id real
    document.getElementById('user_id').addEventListener('blur', function() {
      var userName = this.value.trim();
      if (!userName) return;

      var xhr = new XMLHttpRequest();
      xhr.open("POST", "loginqr.php", true);
      xhr.setRequestHeader("Content-Type", "application/x-www-form-urlencoded");
      xhr.onreadystatechange = function() {
        if (xhr.readyState === 4) {
          if (xhr.status === 200) {
            try {
              var resp = JSON.parse(xhr.responseText);
              if (resp.status === 'success') {
                // Preenche o campo hidden com o user_id
                document.getElementById('hidden_user_id').value = resp.user_id;
              } else {
                // user not found ou erro
                document.getElementById('hidden_user_id').value = '';
                // Se quiser, pode exibir um alert
                // alert("Usuário não encontrado!");
              }
            } catch(e) {
              console.log(e);
            }
          }
        }
      };
      xhr.send("action=fetch_user_id&username=" + encodeURIComponent(userName));
    });
  </script>

  <!-- Ícones Font Awesome -->
  <link
    rel="stylesheet"
    href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/4.7.0/css/font-awesome.min.css"
  >
</body>
</html>
